package Parp::Utils;

use strict;

use Fcntl qw(:DEFAULT :flock);

use MyFilter qw(%CONFIG);
use Parp::Options qw(opt);

use base qw(Exporter);
use vars qw(@EXPORT_OK);
@EXPORT_OK = qw(init_log log_to_file vprint log_rule
                fatal check_file_dir 
                global_lock global_unlock
                i2month month2i);

sub init_log {
  die "No log_file specified in %CONFIG\n" unless $CONFIG{log_file};
  check_file_dir($CONFIG{log_file});
  open(LOG, ">>$CONFIG{log_file}")
    or die "Couldn't open log file `$CONFIG{log_file}' for writing: $!\n";
}

sub log_to_file (@) {
  my (@msgs) = @_;
  if (opt('test_run')) {
    print @msgs;
  }
  else {
    if (fileno LOG) {
      print LOG @msgs;
    }
    else {
      open(BROKEN, ">>/home/adam/mail/.parp.broken") or die "argh!!! $!";
      print BROKEN "fileno(LOG) undef:\n@msgs\n";
      close(BROKEN);
    }
  }
}

my $WIDTH = 78;

sub log_rule {
  log_to_file $_[0] x $WIDTH, "\n";
}

sub vprint (@) {
  # Deal with messages to be printed/logged when user specifies -v
  my (@msgs) = @_;
  if (opt('verbose')) {
    print @msgs unless opt('test_run');
  }
  log_to_file @msgs;
}

sub check_file_dir {
  my ($file) = @_;

  my $umask = 0700;

  my ($dir) = $file =~ m!(.*)/!;
  return unless $dir;

  unless (-d $dir) {
    mkpath([$dir], 0, $umask);
    vprint sprintf "Created directory $dir with umask %04o.\n", $umask
      if fileno(LOG);
  }
}

sub global_lock {
  die "No lock_file specified in %CONFIG\n" unless $CONFIG{lock_file};
  check_file_dir($CONFIG{lock_file});

  if (! -e $CONFIG{lock_file}) {
    unless (open(LOCK, ">$CONFIG{lock_file}")) {
      fatal("Couldn't create lock file $CONFIG{lock_file}: $!");
      exit 3;
    }
  }
  else {
    unless (open(LOCK, $CONFIG{lock_file})) {
      fatal("Couldn't open lock file $CONFIG{lock_file}: $!");
      exit 4;
    }
  }

  my $wait = 0;
  until (flock LOCK, LOCK_EX | LOCK_NB) {
    vprint "\n" if $wait;
    vprint "Waiting for exclusive lock on $CONFIG{lock_file} ... ";
    $wait++;
    sleep 3;
  }
  vprint "got it!\n" if $wait;
}

sub global_unlock {
  # Don't do anything; LOCK_UN introduces races.
}

##############################################################################
#
# Miscellaneous routines
#

sub fatal {
  my ($error, @context) = @_;

  my $long_message = <<EOF;

*************************************************************************

A fatal error occurred:

$error
EOF

  if (@context) {
    $long_message .= <<EOF;
Context for the error follows below:

-------- 8< -------- 8< --------
EOF

  $long_message .= join '', @context;

  $long_message .= <<EOF;
-------- 8< -------- 8< --------
EOF
  }
  else {
    $long_message .= <<EOF;

There was no context given for the error.
EOF
  }

  $long_message .= <<EOF;

*************************************************************************
EOF

  log_to_file $long_message if fileno LOG;
  print $long_message if opt('verbose');

  warn "parp: $error (fatal)" .
       ((fileno LOG) ? "; see log file $CONFIG{log_file} for context.\n"
                    : ".\n");

  if ($CONFIG{fatals_folder}) {
    my $date = localtime();
    my ($username, $realname) = (getpwuid $<)[0, 6];
    my $report = <<"End of report";
From $username\@localhost $date
From: "parp e-mail filter" <$username\@localhost>
To: "$realname" <$username\@localhost>
Date: $date
Subject: parp experienced a fatal error

$long_message

End of report

    if (open FATALS, ">>$CONFIG{fatals_folder}") {
      print FATALS $report;
      close FATALS;
    }
  }
}

my @months = qw/Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec/;
my %month2i = map { $months[$_] => $_ } 0 .. 11;
my %i2month = map { $_ => $months[$_] } 0 .. 11;

sub month2i { $month2i{$_[0]} }
sub i2month { $i2month{$_[0]} }

END {
  close LOG if fileno LOG;
}

1;

