package Parp::Folders;

=head1 NAME

Parp::Folders

=head1 DESCRIPTION

This is essentially a cache of file handles for output folders.

=cut

use strict;

use Parp::Options qw(opt);
use Parp::Utils qw(vprint check_file_dir);

use base qw(Exporter);
use vars qw(@EXPORT_OK);
@EXPORT_OK = qw(append_to_folder);

my %out_folders = ();

sub append_to_folder {
  my ($file, $mail) = @_;

  my $out = $out_folders{$file} ||= _get_folder_fh($file);

  if (ref($out) eq 'Mail::Folder') {
    $out->append_message($mail->{mail});
  }
  else {
    my $text = $mail->{mail}->as_mbox_string();
    $text =~ s/^Mail-From: /From /;
    $text =~ s/\n+$/\n\n/;
    print $out $text;
  }
}

sub _get_folder_fh {
  my ($file) = @_;
  
  check_file_dir($file);

  if (opt('Mail_Folder')) {
    # Use Mail::Folder
    if (-e $file) {
      vprint "Opening $file for appending ... ";
      my $fh = new Mail::Folder('AUTODETECT', $file);
      vprint "done.\n";
      return $fh;
    }
    else {
      return new Mail::Folder('mbox', $file, Create => 1);
    }
  }
  else {
    # Don't use Mail::Folder
    local *FH;
    unless (open(FH, ">>$file")) {
      fatal("Couldn't open $file for delivery: $!");
      exit 5;
    }
    vprint "Opened $file for appending.\n";
    return *FH;
  }
}

END {
  foreach my $name (keys %out_folders) {
    my $folder = $out_folders{$name};
    if (ref($folder) eq 'Mail::Folder') {
      vprint "Syncing $name ... ";
      $folder->sync();
      vprint "done.\n";

      vprint "Closing $name ... ";
      $folder->close();
      vprint "done.\n";
    }
    else {
      close($folder);
    }
  }
}

1;
