package Parp::Utils;

use strict;
use warnings;

use Fcntl qw(:seek);
use File::Path;

use Parp::Config qw(config);
use Parp::Options qw(opt);

use base 'Exporter';
our @EXPORT_OK = qw(init_log close_log 
                    log_to_file vprint log_rule log_header log_mode
                    error fatal check_file_dir 
                    lock_ident unlock_ident lock_file unlock_file
                    i2month month2i);

sub init_log {
  my $log = config->log_file();
  die "No log_file specified in config\n" unless $log;
  check_file_dir($log);
  open(LOG, ">>$log") or die "Couldn't open log file `$log' for writing: $!\n";
  select((select(LOG), $| = 1)[0]);
}

sub close_log {
  close LOG if fileno LOG;
}

sub log_to_file (@) {
  my (@msgs) = @_;
  if (opt('test_run')) {
    print @msgs;
  }
  else {
    if (fileno LOG) {
      seek LOG, 0, SEEK_END;
      print LOG @msgs;
    }
    else {
      open(BROKEN, ">>$ENV{HOME}/mail/.parp.broken") or die "argh!!! $!";
      print BROKEN "fileno(LOG) undef:\n@msgs\n";
      close(BROKEN);
    }
  }
}

my $WIDTH = 78;

sub log_mode {
  my $msg = join '', @_;
  $msg =~ s/^/### /gm;
  log_to_file "###\n$msg\n###\n\n";
}

sub log_rule {
  log_to_file $_[0] x $WIDTH, "\n";
}

sub vprint (@) {
  # Deal with messages to be printed/logged when user specifies -v
  my (@msgs) = @_;
  if (opt('verbose')) {
    print @msgs unless opt('test_run');
  }
  log_to_file @msgs;
}

sub check_file_dir {
  my ($file) = @_;

  my $umask = 0700;

  my ($dir) = $file =~ m!(.*)/!;
  return unless $dir;

  unless (-d $dir) {
    mkpath([$dir], 0, $umask);
    vprint sprintf "Created directory $dir with umask %04o.\n", $umask
      if fileno(LOG);
  }
}

sub log_header {
  my ($m) = @_;
  
  vprint "Parp-ID: $m->{parp_id}\n";

  log_to_file <<EOF;
From: $m->{from}
To: $m->{to}
EOF

  log_to_file "Cc: $m->{cc}\n" if $m->{cc};
  log_to_file "Subject: $m->{subject}\n";

  if ($m->{parp_id}) {
  }
  elsif (1) {
    error("Parp-ID not defined",
          "\$m:\n", Dumper($m),
         );
  }
  # the following cases should never happen
  elsif ($m->{id} && $m->{id} ne '<>') {
    vprint "Message-ID: $m->{id}\n";
  }
  elsif ($m->{date}) {
    vprint "Date: $m->{date}\n";
  }
  elsif ($m->{subject}) {
    vprint "Subject: $m->{subject}\n";
  }
  elsif ($m->{from}) {
    vprint "From: $m->{from}\n";
  }
  else {
    vprint "From $m->{env_from}\n";
  }
  log_to_file "\n";
}

sub fatal {
  my $exit_code = shift;
  error(@_);
  exit $exit_code;
}

sub error {
  my ($error, @context) = @_;

  my $long_message = <<EOF;

*************************************************************************

An error occurred:

$error
EOF

  if (@context) {
    $long_message .= <<EOF;
Context for the error follows below:

-------- 8< -------- 8< --------
EOF

  $long_message .= join '', @context;

  $long_message .= <<EOF;
-------- 8< -------- 8< --------
EOF
  }
  else {
    $long_message .= <<EOF;

There was no context given for the error.
EOF
  }

  $long_message .= <<EOF;

*************************************************************************
EOF

  log_to_file $long_message if fileno LOG;
  print $long_message if opt('verbose');

  my $log = config->log_file();
  warn "parp: $error" .
       ((@context && fileno LOG)
          ? "; see log file $log for context.\n" : ".\n");

  if (my $errors = config->errors_folder()) {
    my $date = localtime();
    my ($username, $realname) = (getpwuid $<)[0, 6];
    my $report = <<"End of report";
From $username\@localhost $date
From: "parp e-mail filter" <$username\@localhost>
To: "$realname" <$username\@localhost>
Date: $date
Subject: parp experienced an error

$long_message

End of report

    if (open ERRORS, ">>$errors") {
      print ERRORS $report;
      close ERRORS;
    }
  }
}

my @months = qw/Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec/;
my %month2i = map { $months[$_] => $_ } 0 .. 11;
my %i2month = map { $_ => $months[$_] } 0 .. 11;

sub month2i { $month2i{$_[0]} }
sub i2month { $i2month{$_[0]} }

END {
  close_log();
}

1;

