package Parp::Folders;

=head1 NAME

Parp::Folders

=head1 DESCRIPTION

This is essentially a cache for the Mail::Box::Manager object
and output folders.

=cut

use strict;
use warnings;

use Fcntl qw(:seek);
use POSIX qw(strftime);

use Mail::Box::Manager;
use Parp::Utils qw(vprint check_file_dir error fatal);
use Parp::Locking;

use base qw(Exporter);
our @EXPORT_OK = qw(append_to_folder folder_substs);

my %out_folders = ();
my $mgr = Mail::Box::Manager->new(default_folder_type => 'mbox');

sub folder_substs {
  my ($folder) = @_;
  my @now = localtime();
  my $quarter = int($now[4]/3 + 1);
  $folder = strftime($folder, @now);
  $folder =~ s/%q/$quarter/g;
  return $folder;
}

sub append_to_folder {
  my ($file, $mail) = @_;

  my $message = $mail->{mail};

  if (eval { $message->isa('Mail::Box::Message') }) {
    eval {
      $mgr->appendMessages($file, $message, access => 'rw', create => 1);
    };
    error("Mail::Box::Manager::appendMessages() failed: $@") if $@;
  }
  else {
    my $out = $out_folders{$file} ||= _get_folder_fh($file);
    my $text = $message->as_mbox_string();
    $text =~ s/^Mail-From: /From /;
    $text =~ s/\n+$/\n\n/;
    lock_file($file, $out);
#    print $out $message->fromLine() if $message->can('fromLine');
    seek $out, 0, SEEK_END;
    print $out $text;
    unlock_file($file, $out);
  }
}

sub _get_folder_fh {
  my ($file) = @_;
  
  check_file_dir($file);

  local *FH;
  unless (open(FH, ">>$file")) {
    fatal(5, "Couldn't open $file for delivery: $!");
  }
  vprint "Opened $file for appending.\n";
  return *FH;
}

END {
  foreach my $name (keys %out_folders) {
    close($out_folders{$name});
  }
}

1;
