package Parp::Filter;

use strict;
use warnings;

use Parp::Config  qw(config);
use Parp::Options qw(opt);
use Parp::Utils   qw(vprint log_to_file log_header log_rule);

use base 'Exporter';
our @EXPORT_OK = qw(filter);

# This subroutine is the heart of the filtering strategy.  It places
# the mail currently being filtered into one of the following
# categories:
# 
#   IS_SPAM    -- spam
#   TO_MAIN    -- good mail destined for main inbox
#   TO_AUX     -- good mail destined for auxiliary inboxes
#   SPECIAL    -- leave subroutines to do their own thang

sub categorize {
  my ($self, $m) = @_;

  if ($m->was_to_old_addresses()) {
    $m->reject_mail('was to old address');
    $m->deliver_to_inbox('old_addresses');
    return 'TO_AUX';
  }

  # Ignore real lusers.
  return 'IS_SPAM' if $m->has_spam_from_addresses();

  # Allow config file to deal with special cases in its own way.
  my $rv = $self->is_special($m);
  return $rv if $rv;

  return 'TO_MAIN' if $self->is_from_daemon($m);

  # Any good signs which indicate that the mail should definitely
  # NOT be treated as junk?
  my $grace = $m->is_passworded()        ||
              $m->is_from_good_person()  ||
              $m->is_from_good_domain()  ||
              $m->has_good_headers();

  return 'IS_SPAM'
    if ! $grace &&
       ($m->has_spam_headers()          ||
        $m->has_spam_domains_anywhere() ||
        $m->has_spam_content());

  if ($self->is_list_mail($m)) {
    $m->{complain} = 0;
    return 'TO_AUX';
  }

  return 'TO_MAIN' if $grace;

  # We put this one after the check for list mail, because on average,
  # mail from lists tends to be lower grade than personal mail.
  return 'IS_SPAM' if $m->has_suspicious_headers();

  return 'IS_SPAM' if ! $m->for_me();

  $m->accept_mail('passed all tests');
  return 'TO_MAIN';
}

=head2 process_mail()

This is where all the fun starts.  Takes a Mail::Internet as
parameter, and filters it.

=cut

sub process_mail {
  my ($self, $mail) = @_;

  my $folder;
  $folder = $mail->{parp_foldername} if $mail->{parp_foldername};

  if (! $mail) {
    error('message parsing failed',
          "\$folder:\n", Dumper($folder),
          "\n",
          "\$mail:\n", Dumper($mail),          
         );
    return 0;
  }

  my $m = Mail::Filterable->new($mail);

  log_header($m);

  return $m->extract_friends($folder) if opt('extract_friends');

  my $wrong_class = opt('wrong_class');
  if (opt('no_dups') && ! $wrong_class && $m->is_duplicate()) {
    $m->reject_mail('was duplicate by message id');
#   $m->deliver_to_inbox('duplicates');
    $m->{backup} = 0;
    return 'IS_DUPLICATE';
  }

  # FIXME: There could be more than one X-Loop header.
  if (($m->{header}->get('X-Loop') || '') eq config->loop_value() and
     ! $wrong_class) {
    $m->accept_mail('looped');
    return 'LOOPED';
  }

  $m->{filter_category} = $self->categorize($m);

  if (! $wrong_class) {
    $m->parse_received_headers();

    if ($m->{recvd_parses_failed}) {
      $m->{header}->add('X-Parp-Received-Parse',
                        $m->{recvd_parses_failed}
                        . ' header'
                        . ($m->{recvd_parses_failed} > 1 ? 's' : '')
                        . ' failed parse');
      if ($m->{filter_category} eq 'IS_SPAM') {
        $m->deliver_to('spam_recvds');
      }
      else {
        vprint $m->{recvd_parses_out};
        $m->deliver_to('bad_recvds');
      }
    }

    if ($m->{filter_category} eq 'TO_MAIN') {
      $self->default_deliver($m);
    }
    elsif ($m->{filter_category} eq 'IS_SPAM') {
      if ($m->{complain}) {
        # TODO: write and send a rude letter to relevant abuse@foo.com address.
        log_to_file "Would complain\n";
      }
    }
    elsif ($m->{filter_category} eq 'TO_AUX') {
      # list mail; already delivered to primary target
      #  - maybe back up though
      $m->maybe_backup();
    }
    elsif ($m->{filter_category} eq 'IS_SPECIAL') {
      # special case mail; already delivered to primary target
      #  - maybe back up though
      $m->maybe_backup();
    }
    else {
      die "Oh dear.";
    }
  }
  else {
    # The user's telling us that the filter_category we've just
    # calculated is wrong.
    if ($m->{filter_category} eq 'IS_SPAM') {
      vprint "Reclassification: was incorrectly identified as spam\n";
      $m->{filter_category} = 'UNKNOWN_NOT_SPAM';
    }
    elsif ($m->{filter_category} ne 'IS_SPAM') {
      vprint "Reclassification: was incorrectly identified as bona-fide\n";
      $m->{filter_category} = 'IS_SPAM';
    }
  }

  log_rule('-');

  return $m->{filter_category};
}

sub default_deliver {
  my $self = shift;
  my ($m) = @_;
  $m->deliver_to_inbox(config->main_folder());
}

sub is_from_daemon {
  my $self = shift;
  my $m = shift;

  # standard daemon errors
  if (
      ($m->{from} =~ /
                      Mail Delivery (Subsystem|System)    |
                      MAILER-DAEMON                       |
                      MESSAGE_ERROR_HANDLER               |
                      postmaster\@
                     /ix or
       $m->{env_from} =~ /MAILER-DAEMON/
      )
      and
      $m->{subject} =~ /(
                         Undeliverable                                    |
                         Returned\ mail:\ User unknown|subscri(be|ption)  |
                         delivery\ fail(ed|ure)                           |
                         failure\ notice                                  |
                         returning\ (mail|message)
                        )/ix
     )
  {
    $m->accept_mail('from daemon', "subject contained `$1'");
    return 1;
  }

  return 0;
}

sub new { shift } # monadic class, no state needed

# This is naughty but convenient.
my $user_filter = "$ENV{HOME}/.parp/Filter.pm";
if (-e $user_filter) {
  eval { require $user_filter };
  die $@ if $@;
}
my $filter = Parp::Filter::User->can('new') ?
               Parp::Filter::User->new() : Parp::Filter->new();
sub filter { $filter }

1;
