package Parp::Config;

=head1 NAME

Parp::Config - base class for Parp config

=cut

use strict;
use warnings;

use base 'Exporter';
our @EXPORT_OK = qw(config);

# Directory where delivery folders who do not have a `/' specified live
sub mail_dir  { "$ENV{HOME}/mail"                }
sub inbox_dir { shift->mail_dir() . '/inboxes'   }
sub inbox     { shift->inbox_dir() . '/' . shift }

# Directory containing inboxes

sub main_folder   { 'MAIN'                                   }
sub backup_folder { shift->mail_dir() . "/backup"            }

# Folder where reports of parp errors are delivered
sub errors_folder { shift->inbox("parp-errors")              }

sub log_file      { shift->mail_dir() . "/.parp.log"         }
sub lock_dir      { shift->mail_dir() . "/.parp.lock"        }

# Filename for DBM database of your friends
sub friends_db    { shift->mail_dir() . "/.friends.db"       }

# Cache containing Message-ID's of recent messages, used to eliminate duplicates
sub id_cache      { shift->mail_dir() . "/.parp.msgid.cache" } 
# Maximum number of Message-IDs in cache
sub max_cache_ids { 200 }

# Value of X-Loop header set to avoid mail loops.
sub loop_value {
  my ($login, $username) = (getpwuid($<))[0, 6];
  return $username || $login;
}

# If the subject line or header corresponding to
# the password_header value below match the following
# password, the mail is given special consideration
sub password        { 'fragglerock' }
sub password_header { 'X-Password'  }

# Threshold number of recipients before an e-mail
# is considered as spam
sub max_recipients { 100 }
  
# If the body has more than ...
sub max_forwards_lines { 50 } 
# ... lines containing more than ...
sub max_forwards { 15 } 
# '>' quotation marks, I don't want to see it.

# If the body matches this more than the given amount, I don't
# want to see the e-mail.  Be very careful what you put in here!
sub max_quite_bad_words { 8 } 
sub max_unique_quite_bad_words { 5 } 

sub max_like_me { 5 }
sub like_me     {   }

# If the body, or any nested multiparts have a Content-Type
# matching this, I don't want to see it.
sub bad_content_type { qr/charset="(ks_|.*-jp)/ } 

sub me                { die "Config::me() not overridden"        }
sub old_me            { qr/nasty hack - this should never match/ }
sub decoys            { qr/nasty hack - this should never match/ }
sub good_domains      { qr/nasty hack - this should never match/ }
sub subject_buzzwords { qr/\b [Pp]erl | parp \b/x                }

{
  (my $dir = $INC{'Parp/Config.pm'}) =~ s!\.pm$!!; # hope this works ...
  sub regexp_dir { $dir }
}

sub lines_from_file {
  my ($self, $file) = @_;
  open(FILE, $file) or die "Couldn't open $file: $!\n";
  my @lines = ();
  while (<FILE>) {
    next if /^\s*#/;
    s/^\s*(.*?)\s*$/$1/;
    push @lines, $_ if $_;
  }
  close(FILE);
  return @lines;
}

sub regexp_from_file {
  my ($self, $file, $options, @extra_regexps) = @_;
  my @lines = $self->lines_from_file($self->regexp_dir() . '/' . $file);
  my $re = join '|', @lines, @extra_regexps;
  $re =~ s!/!\\/!g;
  my $regexp = eval "qr/($re)/$options";
  die $@ if $@;
# warn "$file is regexp m[$regexp]\n";
  return $regexp;
} 

sub bad_from        { shift->regexp_from_file('bad_from',        'i' ) }
sub bad_to          { shift->regexp_from_file('bad_to',          'i' ) }
sub bad_subjects    { shift->regexp_from_file('bad_subjects',    'ix') }
sub bad_words       { shift->regexp_from_file('bad_words',       'ix') }
sub bad_origins     { shift->regexp_from_file('bad_origins',     'ix') }
sub quite_bad_words { shift->regexp_from_file('quite_bad_words', 'ix') }
sub very_bad_words  { shift->regexp_from_file('very_bad_words',  'ix') }

sub new { shift } # monadic class, no state needed

# This is naughty but convenient.
my $user_config = "$ENV{HOME}/.parp/Config.pm";
if (-e $user_config) {
  eval { require $user_config };
  die $@ if $@;
}
my $config = Parp::Config::User->can('new') ?
               Parp::Config::User->new() : Parp::Config->new();
sub config { $config }

1;
